package rabbit.proxy;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import rabbit.io.HandlerRegistration;
import rabbit.io.SocketHandler;
import rabbit.util.Logger;

/** A class that handles a selector. 
 *
 * @author <a href="mailto:robo@khelekore.org">Robert Olofsson</a>
 */
public class SelectorRunner implements Runnable {

    /** The selector in use. */
    private Selector selector = null;

    /** Is this proxy running. */
    private boolean running = false;

    /** The queue to get back on the main thread. */
    private Object returnedTasksLock = new Object ();
    private List<Runnable> returnedTasks1 = new ArrayList<Runnable> ();
    private List<Runnable> returnedTasks2 = new ArrayList<Runnable> ();

    /** The executor service. */
    private TaskRunner tr;

    private Logger logger;

    public SelectorRunner (TaskRunner tr, Logger logger) throws IOException {
	this.tr = tr;
	this.logger = logger;
	selector = Selector.open ();
    }

    public void start () {
	running = true;	
    }

    public void stop () {
	running = false;
	try {
	    if (selector != null) {
		selector.close ();
		selector = null;
	    }
	} catch (IOException e) {
	    logger.logFatal ("Failed to close selector " + 
			     getStackTrace (e));
	}
    }

    public Selector getSelector () {
	return selector;
    }

    public void runSelectorTask (Runnable r) {
	synchronized (returnedTasksLock) {
	    returnedTasks1.add (r);
	}
	synchronized (this) {
	    // ensure that we do not get NPE 
	    selector.wakeup ();
	}
    }

    public void run () {
	while (running) {
	    while (running && !selector.isOpen ()) {
		try {
		    // wait for reconfigure
		    Thread.sleep (2 * 1000);
		} catch (InterruptedException e) {
		    // ignore
		}
	    }
	    try {
		selector.select (10 * 1000);
		if (selector.isOpen ()) {
		    cancelTimeouts ();
		    handleSelects ();
		    runReturnedTasks ();
		}
	    } catch (IOException e) {
		logger.logError ("Failed to accept, " + 
				 "trying to restart serversocket: " + e +
				 "\n" + getStackTrace (e));
		synchronized (this) {
		    stop ();
		    start ();
		}
	    } catch (Exception e) {
		logger.logError ("Unknown error: " + e + 
				 " attemting to ignore\n" + 
				 getStackTrace (e));
	    }
	}
    }

    private String getStackTrace (Throwable t) {
    	StringWriter sw = new StringWriter ();
	PrintWriter ps = new PrintWriter (sw);
	t.printStackTrace (ps);
	return sw.toString ();
    }

    private void cancelTimeouts () throws IOException {
	long now = System.currentTimeMillis ();
	for (SelectionKey sk : selector.keys ()) {
	    Object a = sk.attachment ();
	    if (a instanceof String) {
		// ignore, this is used for status.
	    } else {
		HandlerRegistration hr = (HandlerRegistration)a;
		if (hr != null && hr.isExpired (now, 60 * 1000)) {
		    cancelKeyAndCloseChannel (sk);
		    hr.timeout ();
		}
	    }
	}
    }
    
    /** Close down a client that has timed out. 
     */
    private void cancelKeyAndCloseChannel (SelectionKey sk) {
	sk.cancel ();
	try {
	    SocketChannel sc = (SocketChannel)sk.channel ();
	    sc.close ();
	} catch (IOException e) {
	    logger.logError ("failed to shutdown and close socket: " + e);
	}
    }
    
    private void handleSelects () throws IOException {
	Set<SelectionKey> selected = selector.selectedKeys ();
	for (Iterator<SelectionKey> i = selected.iterator (); i.hasNext (); ) {
	    SelectionKey sk = i.next ();
	    Object a = sk.attachment ();
	    if (a != null && a instanceof HandlerRegistration) {
		HandlerRegistration hr = (HandlerRegistration)a;
		if (sk.isValid ()) {
		    SocketHandler sh = hr.getHandler (sk);
		    if (sh != null)
			handle (sk, sh);
		} else {
		    cancelKeyAndCloseChannel (sk);
		    hr.timeout ();
		}
	    } else if (a == null) {
		logger.logWarn ("No handler for:" + sk);
	    } else {
		// Ok, something is very bad here, try to shutdown the channel
		// and hope that we handle it ok elsewhere...
		logger.logError ("Bad handler for:" + sk + ": " + a);
		sk.cancel ();
		sk.channel ().close ();
	    }
	}
	selected.clear ();
    }

    private void handle (SelectionKey sk, SocketHandler handler) {
	if (handler.useSeparateThread ()) {
	    // need to cancel so that we do not get multiple selects...
	    sk.cancel (); 
	    tr.runThreadTask (handler);
	} else {
	    handler.run ();
	}
    }

    private void runReturnedTasks () {
	synchronized (returnedTasksLock) {
	    List<Runnable> toRun = returnedTasks1;
	    returnedTasks1 = returnedTasks2;
	    returnedTasks2 = toRun;
	}
	int s = returnedTasks2.size ();
	for (int i = 0; i < s; i++)
	    returnedTasks2.get (i).run ();
	returnedTasks2.clear ();
    }
}
