package rabbit.handler;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import rabbit.filter.HtmlFilter;
import rabbit.filter.HtmlFilterFactory;
import rabbit.html.HtmlBlock;
import rabbit.html.HtmlParseException;
import rabbit.html.HtmlParser;
import rabbit.http.HttpHeader;
import rabbit.io.BufferHandle;
import rabbit.io.SimpleBufferHandle;
import rabbit.proxy.Connection;
import rabbit.proxy.TrafficLoggerHandler;
import rabbit.util.Logger;
import rabbit.util.SProperties;
import rabbit.zip.GZipUnpackListener;
import rabbit.zip.GZipUnpacker;

/** This handler filters out unwanted html features.
 *
 * @author <a href="mailto:robo@khelekore.org">Robert Olofsson</a>
 */
public class FilterHandler extends GZipHandler {
    private SProperties config = new SProperties ();
    private List<HtmlFilterFactory> filterClasses = 
    new ArrayList<HtmlFilterFactory> ();
    private boolean repack = false;

    private List<HtmlFilter> filters; 
    private HtmlParser parser;
    private byte[] restBlock = null;
    private boolean sendingRest = false;
    private Iterator<ByteBuffer> sendBlocks = null;

    private GZipUnpacker gzu = null;
    private GZListener gzListener = null;

    // For creating the factory.
    public FilterHandler () {	
    }

    /** Create a new FilterHandler for the given request.
     * @param con the Connection handling the request.
     * @param request the actual request made.
     * @param clientHandle the client side buffer handle.
     * @param response the actual response.
     * @param content the resource.
     * @param mayCache May we cache this request? 
     * @param mayFilter May we filter this request?
     * @param size the size of the data beeing handled.
     * @param compress if we want this handler to compress or not.
     */
    public FilterHandler (Connection con, TrafficLoggerHandler tlh, 
			  HttpHeader request, BufferHandle clientHandle,
			  HttpHeader response, ResourceSource content, 
			  boolean mayCache, boolean mayFilter, long size, 
			  boolean compress, boolean repack,
			  List<HtmlFilterFactory> filterClasses) {
	super (con, tlh, request, clientHandle, response, content, 
	       mayCache, mayFilter, size, compress);
	this.repack = repack;
	this.filterClasses = filterClasses;
    }

    @Override 
    protected void setupHandler () {
	String ce = response.getHeader ("Content-Encoding");
	if (repack && ce != null) {
	    ce = ce.toLowerCase ();
	    if (ce.equals ("gzip")) {
		gzListener = new GZListener ();
		gzu = new GZipUnpacker (gzListener);		
	    }
	}

	super.setupHandler ();
	if (mayFilter) {
	    response.removeHeader ("Content-Length");
	    parser = new HtmlParser ();
	    filters = initFilters ();
	}
    }

    @Override 
    protected boolean willCompress (HttpHeader request) {
	return gzu != null || super.willCompress (request);
    }

    private class GZListener implements GZipUnpackListener {
	private byte[] buffer;
	public void unpacked (byte[] buf, int off, int len) {
	    handleArray (buf, off, len);
	}
	
	public void dataUnpacked () {
	    // do not really care...
	}
	
	public void finished () {
	    gzu = null;
	    finishData ();
	}

	public byte[] getBuffer () {
	    if (buffer == null)
		buffer = new byte[4096];
	    return buffer;
	}

	public void returnBuffer (byte[] buf) {
	    // do not really care...
	}

	public void failed (Exception e) {
	    FilterHandler.this.failed (e);
	}
    }
    
    @Override
    public Handler getNewInstance (Connection con, TrafficLoggerHandler tlh,
				   HttpHeader header, BufferHandle bufHandle, 
				   HttpHeader webHeader, 
				   ResourceSource content, boolean mayCache, 
				   boolean mayFilter, long size) {
	FilterHandler h = 
	    new FilterHandler (con, tlh, header, bufHandle, webHeader, 
			       content, mayCache, mayFilter, size, 
			       compress, repack, filterClasses);
	h.setupHandler ();
	return h;
    }

    @Override
    protected void writeDataToGZipper (byte[] arr) {
	forwardArrayToHandler (arr, 0, arr.length);
    }

    @Override
    protected void modifyBuffer (BufferHandle bufHandle) {
	if (!mayFilter) {
	    super.modifyBuffer (bufHandle);
	    return;
	}
	ByteBuffer buf = bufHandle.getBuffer ();
	byte[] arr; 
	if (buf.isDirect ()) {
	    arr = new byte[buf.remaining ()];
	    buf.get (arr);
	} else {
	    arr = buf.array ();
	}
	bufHandle.possiblyFlush ();
	forwardArrayToHandler (arr, 0, arr.length);
    }

    private void forwardArrayToHandler (byte[] arr, int off, int len) {
	if (gzu != null)
	    gzu.setInput (arr, 0, arr.length);
	else
	    handleArray (arr, 0, arr.length);
    }

    private void handleArray (byte[] arr, int off, int len) {
	if (restBlock != null) {
	    int rs = restBlock.length;
	    int newLen = len + rs;
	    byte[] buf = new byte[newLen];
	    System.arraycopy (restBlock, 0, buf, 0, rs);
	    System.arraycopy (arr, off, buf, rs, len);
	    arr = buf;
	    off = 0;
	    len = newLen;
	}
	parser.setText (arr, off, len);
	HtmlBlock currentBlock = null;
	try {
	    currentBlock = parser.parse ();
	    for (HtmlFilter hf : filters) {
		hf.filterHtml (currentBlock);
		if (!hf.isCacheable ()) {
		    mayCache = false;
		    removeCache ();
		}
	    }
	    
	    List<ByteBuffer> ls = currentBlock.getBlocks (); 
	    if (currentBlock != null && currentBlock.restSize () > 0) {
		// since the unpacking buffer is re used we need to store the 
		// rest in a separate buffer.
		restBlock = new byte[currentBlock.restSize ()];
		currentBlock.insertRest (restBlock);
	    } else {
		restBlock = null;
	    }
	    sendBlocks = ls.iterator ();
	    if (sendBlocks.hasNext ()) {
		sendBlockBuffers ();
	    } else {
		// no more blocks so wait for more data, either from 
		// gzip or the net
		blockSent ();
	    }
	} catch (HtmlParseException e) {
	    getLogger ().logInfo ("Bad HTML: " + e.toString ());
	    // out.write (arr);
	    currentBlock = null;
	}
    }

    public void blockSent () {
	if (sendingRest) {
	    super.finishData ();
	} else if (sendBlocks != null && sendBlocks.hasNext ()) {
	    sendBlockBuffers (); 
	} else if (gzu != null && !gzu.needsInput ()) {
	    gzu.handleCurrentData ();
	} else {
	    super.blockSent ();
	}
    }

    private void sendBlockBuffers () {
	ByteBuffer buf = sendBlocks.next ();
	SimpleBufferHandle bh = new SimpleBufferHandle (buf);
	send (bh);
    }

    @Override
    protected void finishData ()  {
	if (restBlock != null && restBlock.length > 0) {
	    ByteBuffer buf = ByteBuffer.wrap (restBlock);
	    SimpleBufferHandle bh = new SimpleBufferHandle (buf);
	    restBlock = null;
	    sendingRest = true;
	    send (bh);
	} else {
	    super.finishData ();
	}
    }

    /** Initialize the filter we are using.
     * @return a List of HtmlFilters.
     */
    private List<HtmlFilter> initFilters () {
	int fsize = filterClasses.size ();
	List<HtmlFilter> fl = new ArrayList<HtmlFilter> (fsize);	

	for (int i = 0; i < fsize; i++) {
	    HtmlFilterFactory hff = filterClasses.get (i);
	    fl.add (hff.newFilter (con, request, response));
	}
	return fl;
    }

    /** Setup this class.
     * @param prop the properties of this class.
     */
    @Override public void setup (Logger logger, SProperties prop) {
	super.setup (logger, prop);
	config = prop;
	String rp = config.getProperty ("repack", "false");
	repack = Boolean.parseBoolean (rp);
	String fs = config.getProperty ("filters", "");
	if ("".equals (fs))
	    return;
	String[] names = fs.split (",");
	for (String classname : names) {
	    try {
		Class<? extends HtmlFilterFactory> cls = 
		    Class.forName (classname).
		    asSubclass (HtmlFilterFactory.class);
		filterClasses.add (cls.newInstance ());
	    } catch (ClassNotFoundException e) {
		logger.logWarn ("Could not find filter: '" + classname + "'");
	    } catch (InstantiationException e) {
		logger.logWarn ("Could not instanciate class: '" + 
				classname + "' " + e);
	    } catch (IllegalAccessException e) {
		logger.logWarn ("Could not get constructor for: '" + 
				classname + "' " + e);
	    }
	}
    }
}
